"use strict";
(() => {
  // src/extension/config.ts
  if (false) {
    throw new Error("__BACKEND_URL__ must be defined at build time");
  }
  var BACKEND_URL = "https://sanitycheck-production.up.railway.app";

  // src/extension/debug.ts
  var DEBUG_ENABLED = false;
  var EXTENSION_VERSION = "1.2.0";
  var DEBUG_SERVER_URL = `${BACKEND_URL}/debug/log`;
  var logQueue = [];
  var MAX_QUEUE_SIZE = 100;
  function sanitizeData(data) {
    if (!data) return null;
    try {
      const seen = /* @__PURE__ */ new WeakSet();
      const sanitized = JSON.parse(JSON.stringify(data, (_key, value) => {
        if (typeof value === "function") {
          return "[Function]";
        }
        if (typeof value === "string" && value.length > 1e3) {
          return value.substring(0, 1e3) + "...[truncated]";
        }
        if (typeof value === "object" && value !== null) {
          if (seen.has(value)) {
            return "[Circular]";
          }
          seen.add(value);
        }
        return value;
      }));
      const jsonStr = JSON.stringify(sanitized);
      if (jsonStr.length > 5e4) {
        return { error: "Data too large to log", size: jsonStr.length };
      }
      return sanitized;
    } catch (error) {
      const err = error;
      return { error: "Failed to sanitize data", errorMessage: err.message };
    }
  }
  async function sendLog(level, message, data = {}, source = "unknown") {
    if (!DEBUG_ENABLED) return;
    const logData = {
      level,
      message,
      data: sanitizeData(data),
      source,
      version: EXTENSION_VERSION,
      timestamp: (/* @__PURE__ */ new Date()).toISOString(),
      url: typeof window !== "undefined" ? window.location?.href ?? "background" : "background",
      userAgent: typeof navigator !== "undefined" ? navigator.userAgent : "unknown"
    };
    try {
      const response = await fetch(DEBUG_SERVER_URL, {
        method: "POST",
        headers: {
          "Content-Type": "application/json"
        },
        body: JSON.stringify(logData),
        signal: AbortSignal.timeout(2e3)
      });
      if (!response.ok) {
        throw new Error(`Server responded with ${response.status}`);
      }
      if (logQueue.length > 0) {
        await flushLogQueue();
      }
    } catch (_error) {
      logQueue.push(logData);
      if (logQueue.length > MAX_QUEUE_SIZE) {
        logQueue.shift();
      }
      console[level === "error" ? "error" : "log"](`[${level}] [${source}]`, message, data);
    }
  }
  async function flushLogQueue() {
    if (logQueue.length === 0) return;
    const logsToSend = [...logQueue];
    logQueue = [];
    for (const logData of logsToSend) {
      try {
        await fetch(DEBUG_SERVER_URL, {
          method: "POST",
          headers: {
            "Content-Type": "application/json"
          },
          body: JSON.stringify(logData),
          signal: AbortSignal.timeout(2e3)
        });
      } catch (_error) {
        logQueue.push(logData);
        if (logQueue.length > MAX_QUEUE_SIZE) {
          logQueue.shift();
        }
        break;
      }
    }
  }
  function logError(message, error, source = "unknown", additionalData = {}) {
    const err = error;
    const errorData = {
      ...additionalData,
      error: {
        name: err?.name,
        message: err?.message,
        stack: err?.stack
      }
    };
    void sendLog("error", message, errorData, source);
  }
  var debug = {
    ENABLED: DEBUG_ENABLED,
    log: (message, data = {}, source = "popup") => {
      void sendLog("info", message, data, source);
    },
    warn: (message, data = {}, source = "popup") => {
      void sendLog("warn", message, data, source);
    },
    error: (message, error, source = "popup", additionalData = {}) => {
      logError(message, error, source, additionalData);
    },
    debug: (message, data = {}, source = "popup") => {
      void sendLog("debug", message, data, source);
    },
    flush: () => flushLogQueue()
  };
  if (typeof window !== "undefined") {
    window.addEventListener("error", (event) => {
      debug.error("Window error", {
        message: event.message,
        filename: event.filename,
        lineno: event.lineno,
        colno: event.colno
      }, "window-error");
    });
    window.addEventListener("unhandledrejection", (event) => {
      debug.error("Unhandled promise rejection", event.reason, "promise-rejection");
    });
    window.debug = debug;
  }
  var debug_default = debug;
})();
