"use strict";
(() => {
  // src/shared/colors.ts
  var baseColors = {
    // Status colors (same in both themes)
    error: "#ef4444",
    errorSubtle: "rgba(239, 68, 68, 0.12)",
    success: "#10b981",
    successSubtle: "rgba(16, 185, 129, 0.12)",
    warning: "#f59e0b",
    warningSubtle: "rgba(245, 158, 11, 0.12)",
    // Severity colors (same in both themes)
    severityCritical: "#ef4444",
    severitySignificant: "#eab308",
    severityMinor: "#737373"
  };
  var colors = {
    ...baseColors,
    // Backgrounds
    bgPrimary: "#0f1117",
    bgSecondary: "#1a1d27",
    bgTertiary: "#242936",
    bgHover: "#2d3344",
    // Text
    textPrimary: "#f0f2f5",
    textSecondary: "#9ca3b0",
    textMuted: "#6b7280",
    // Accent (blue)
    accent: "#60a5fa",
    accentHover: "#93c5fd",
    accentSubtle: "rgba(96, 165, 250, 0.12)",
    // Borders
    border: "rgba(255, 255, 255, 0.08)",
    borderStrong: "rgba(255, 255, 255, 0.12)",
    // Highlight colors (for reasoning gaps)
    highlightCritical: "rgba(239, 68, 68, 0.25)",
    highlightCriticalHover: "rgba(239, 68, 68, 0.4)",
    highlightSignificant: "rgba(234, 179, 8, 0.25)",
    highlightSignificantHover: "rgba(234, 179, 8, 0.4)",
    highlightMinor: "rgba(115, 115, 115, 0.25)",
    highlightMinorHover: "rgba(115, 115, 115, 0.4)",
    highlightDefault: "rgba(96, 165, 250, 0.25)",
    highlightDefaultHover: "rgba(96, 165, 250, 0.4)",
    severityDefault: "#60a5fa"
  };
  var missColors = {
    ...baseColors,
    // Backgrounds (purple-tinted dark)
    bgPrimary: "#1a1520",
    bgSecondary: "#231d2b",
    bgTertiary: "#2d2638",
    bgHover: "#3a3245",
    // Text (lavender-tinted)
    textPrimary: "#f5f0fa",
    textSecondary: "#c4b8d4",
    textMuted: "#8b7fa3",
    // Accent (purple)
    accent: "#c084fc",
    accentHover: "#d8b4fe",
    accentSubtle: "rgba(192, 132, 252, 0.12)",
    // Borders (purple-tinted)
    border: "rgba(192, 132, 252, 0.15)",
    borderStrong: "rgba(192, 132, 252, 0.25)",
    // Highlight colors (purple-shifted for theme consistency)
    highlightCritical: "rgba(239, 68, 68, 0.25)",
    highlightCriticalHover: "rgba(239, 68, 68, 0.4)",
    highlightSignificant: "rgba(234, 179, 8, 0.25)",
    highlightSignificantHover: "rgba(234, 179, 8, 0.4)",
    highlightMinor: "rgba(139, 127, 163, 0.25)",
    highlightMinorHover: "rgba(139, 127, 163, 0.4)",
    highlightDefault: "rgba(192, 132, 252, 0.25)",
    highlightDefaultHover: "rgba(192, 132, 252, 0.4)",
    severityDefault: "#c084fc"
  };
  function generateCssVariables(theme) {
    return `
  --bg-primary: ${theme.bgPrimary};
  --bg-secondary: ${theme.bgSecondary};
  --bg-tertiary: ${theme.bgTertiary};
  --bg-hover: ${theme.bgHover};
  --text-primary: ${theme.textPrimary};
  --text-secondary: ${theme.textSecondary};
  --text-muted: ${theme.textMuted};
  --accent: ${theme.accent};
  --accent-hover: ${theme.accentHover};
  --accent-subtle: ${theme.accentSubtle};
  --error: ${theme.error};
  --error-subtle: ${theme.errorSubtle};
  --success: ${theme.success};
  --success-subtle: ${theme.successSubtle};
  --warning: ${theme.warning};
  --warning-subtle: ${theme.warningSubtle};
  --border: ${theme.border};
  --border-strong: ${theme.borderStrong};
  --severity-critical: ${theme.severityCritical};
  --severity-significant: ${theme.severitySignificant};
  --severity-minor: ${theme.severityMinor};`;
  }
  var cssVariables = `
:root {
${generateCssVariables(colors)}
  --shadow: 0 4px 12px rgba(0, 0, 0, 0.25);
  --radius-sm: 6px;
  --radius: 10px;
  --radius-lg: 14px;
}
`;
  var missCssVariables = `
body.theme-miss {
${generateCssVariables(missColors)}
}
`;
  function missBackgroundCss(imagePath) {
    return `
body.theme-miss {
  background: 
    linear-gradient(
      to bottom,
      rgba(26, 21, 32, 0.85) 0%,
      rgba(26, 21, 32, 0.75) 50%,
      rgba(26, 21, 32, 0.9) 100%
    ),
    url('${imagePath}');
  background-size: cover;
  background-position: center;
  background-attachment: fixed;
}
`;
  }
  var themeCssVariables = cssVariables + missCssVariables + missBackgroundCss("/static/missinfo_bg.jpg");
  var themeCssVariablesExtension = cssVariables + missCssVariables + missBackgroundCss("icons/missinfo_bg.jpg");

  // src/shared/highlight-styles.ts
  var highlightCSS = `
/* ===== CSS Custom Highlight API Styles ===== */
::highlight(logic-checker-critical) {
  background-color: ${colors.highlightCritical};
}

::highlight(logic-checker-significant) {
  background-color: ${colors.highlightSignificant};
}

::highlight(logic-checker-minor) {
  background-color: ${colors.highlightMinor};
}

::highlight(logic-checker-default) {
  background-color: ${colors.highlightDefault};
}

/* ===== Fallback: Span-based Highlight Styles ===== */
.logic-checker-highlight {
  cursor: help;
  position: relative;
  transition: background 0.2s ease;
  border-radius: 2px;
  padding: 1px 2px;
}

.logic-checker-highlight.critical,
.logic-checker-highlight[data-importance="critical"] {
  background: linear-gradient(to bottom, ${colors.highlightCritical} 0%, rgba(239, 68, 68, 0.15) 100%);
}

.logic-checker-highlight.critical:hover,
.logic-checker-highlight[data-importance="critical"]:hover {
  background: ${colors.highlightCriticalHover};
}

.logic-checker-highlight.significant,
.logic-checker-highlight[data-importance="significant"] {
  background: linear-gradient(to bottom, ${colors.highlightSignificant} 0%, rgba(234, 179, 8, 0.15) 100%);
}

.logic-checker-highlight.significant:hover,
.logic-checker-highlight[data-importance="significant"]:hover {
  background: ${colors.highlightSignificantHover};
}

.logic-checker-highlight.minor,
.logic-checker-highlight[data-importance="minor"] {
  background: linear-gradient(to bottom, ${colors.highlightMinor} 0%, rgba(115, 115, 115, 0.15) 100%);
}

.logic-checker-highlight.minor:hover,
.logic-checker-highlight[data-importance="minor"]:hover {
  background: ${colors.highlightMinorHover};
}

.logic-checker-highlight:not(.critical):not(.significant):not(.minor):not([data-importance]) {
  background: linear-gradient(to bottom, ${colors.highlightDefault} 0%, rgba(96, 165, 250, 0.15) 100%);
}

.logic-checker-highlight:not(.critical):not(.significant):not(.minor):not([data-importance]):hover {
  background: ${colors.highlightDefaultHover};
}
`;
  var tooltipCSS = `
.logic-checker-tooltip {
  position: fixed;
  z-index: 2147483647;
  width: max-content;
  max-width: 400px;
  min-width: 280px;
  background: linear-gradient(135deg, #1a1a1a 0%, #0d0d0d 100%);
  border: 1px solid;
  border-radius: 8px;
  padding: 12px 14px;
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5);
  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  font-size: 13px;
  line-height: 1.5;
  color: #f5f5f5;
  pointer-events: none;
  opacity: 0;
  transform: translateY(8px);
  transition: opacity 0.2s ease, transform 0.2s ease;
}

.logic-checker-tooltip.critical {
  border-color: ${colors.severityCritical};
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(239, 68, 68, 0.2);
}

.logic-checker-tooltip.significant {
  border-color: ${colors.severitySignificant};
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(234, 179, 8, 0.2);
}

.logic-checker-tooltip.minor {
  border-color: ${colors.severityMinor};
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(115, 115, 115, 0.2);
}

.logic-checker-tooltip:not(.critical):not(.significant):not(.minor) {
  border-color: ${colors.severityDefault};
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(96, 165, 250, 0.2);
}

.logic-checker-tooltip.visible {
  opacity: 1;
  transform: translateY(0);
}

.logic-checker-tooltip-header {
  margin-bottom: 6px;
}

.logic-checker-tooltip-badge {
  display: inline-block;
  color: #000;
  font-size: 10px;
  font-weight: 600;
  padding: 2px 8px;
  border-radius: 10px;
  text-transform: uppercase;
  letter-spacing: 0.5px;
}

.logic-checker-tooltip.critical .logic-checker-tooltip-badge {
  background: ${colors.severityCritical};
  color: white;
}

.logic-checker-tooltip.significant .logic-checker-tooltip-badge {
  background: ${colors.severitySignificant};
}

.logic-checker-tooltip.minor .logic-checker-tooltip-badge {
  background: ${colors.severityMinor};
  color: white;
}

.logic-checker-tooltip:not(.critical):not(.significant):not(.minor) .logic-checker-tooltip-badge {
  background: ${colors.severityDefault};
}

.logic-checker-tooltip-explanation {
  color: #d4d4d4;
}
`;
  var contentStyles = highlightCSS + "\n" + tooltipCSS;
  var demoHighlightCSS = `
.demo-highlight {
  cursor: help;
  position: relative;
  background: linear-gradient(to bottom, ${colors.highlightSignificant} 0%, rgba(234, 179, 8, 0.15) 100%);
  border-radius: 2px;
  padding: 1px 2px;
  transition: background 0.2s ease;
}

.demo-highlight:hover {
  background: ${colors.highlightSignificantHover};
}

.demo-highlight.critical {
  background: linear-gradient(to bottom, ${colors.highlightCritical} 0%, rgba(239, 68, 68, 0.15) 100%);
}

.demo-highlight.critical:hover {
  background: ${colors.highlightCriticalHover};
}

.demo-highlight.minor {
  background: linear-gradient(to bottom, ${colors.highlightMinor} 0%, rgba(115, 115, 115, 0.15) 100%);
}

.demo-highlight.minor:hover {
  background: ${colors.highlightMinorHover};
}

.demo-tooltip {
  position: absolute;
  z-index: 1000;
  width: max-content;
  max-width: 400px;
  min-width: 280px;
  background: linear-gradient(135deg, #1a1a1a 0%, #0d0d0d 100%);
  border: 1px solid ${colors.severitySignificant};
  border-radius: 8px;
  padding: 12px 14px;
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(234, 179, 8, 0.2);
  font-size: 13px;
  line-height: 1.5;
  color: #f5f5f5;
  pointer-events: none;
  opacity: 0;
  transform: translateY(8px);
  transition: opacity 0.2s ease, transform 0.2s ease;
  left: 0;
  top: 100%;
  margin-top: 8px;
}

.demo-highlight:hover .demo-tooltip {
  opacity: 1;
  transform: translateY(0);
}

.demo-tooltip-header {
  margin-bottom: 6px;
}

.demo-tooltip-badge {
  display: inline-block;
  background: ${colors.severitySignificant};
  color: #000;
  font-size: 10px;
  font-weight: 600;
  padding: 2px 8px;
  border-radius: 10px;
  text-transform: uppercase;
  letter-spacing: 0.5px;
}

.demo-tooltip-explanation {
  color: #d4d4d4;
}
`;

  // src/shared/constants.ts
  var DEFAULT_ANALYSIS_PROMPT = `You help readers notice genuine reasoning problems in articles\u2014things they'd agree are valid weaknesses, even if they agree with the conclusions.

## Your Goal

Surface issues where you're confident it's a real structural flaw AND it matters to the core argument. The cost of a bad objection (annoying the reader, undermining trust) exceeds the cost of missing something. So:

- Only flag things that made you genuinely think "wait, that doesn't follow"
- Try to steelman first\u2014if there's a reasonable interpretation, don't flag
- Ask: would someone who agrees with the article still nod and say "yeah, that's fair"?

Good flags: evidence-conclusion mismatches, load-bearing unstated assumptions, logical leaps that don't follow even charitably.

Bad flags: factual disputes (you might be wrong), nitpicks on tangential points, things that only look wrong if you disagree with the content.

## Output Format

Write your analysis in natural prose. Start with a one-sentence summary of the article's main conclusion. Then describe the central logical weakness (if any) in 3-4 sentences.

For each issue you find, quote the relevant passage (20-60 words) and briefly explain the gap. Keep explanations punchy\u2014the reader should immediately think "oh yeah, that's a leap."

## Rules

- Keep gap explanations brief and immediately recognizable, but still understandable to a university-educated audience. E.g., "Constraints \u2260 impossibility" or "One example doesn't prove a universal"
- Quote exactly from the text
- 1-4 issues typical. Zero is fine if nothing clears the bar. Problematic articles can have up to 6-8.
- Quality over quantity\u2014only flag what you're confident about

ARTICLE:
`;

  // src/extension/settings.ts
  var promptEditor = document.getElementById("prompt-editor");
  var savePromptBtn = document.getElementById("save-prompt");
  var resetPromptBtn = document.getElementById("reset-prompt");
  var statusMessage = document.getElementById("status-message");
  var promptBadge = document.getElementById("prompt-badge");
  var backBtn = document.getElementById("back-btn");
  var themeSanityRadio = document.getElementById("theme-sanity");
  var themeMissRadio = document.getElementById("theme-miss");
  document.addEventListener("DOMContentLoaded", () => {
    void init();
  });
  async function init() {
    const themeData = await chrome.storage.local.get(["theme"]);
    const currentTheme = themeData.theme || "sanity";
    if (currentTheme === "miss") {
      document.body.classList.add("theme-miss");
      document.title = "Miss Information Settings";
      themeMissRadio.checked = true;
    } else {
      document.body.classList.remove("theme-miss");
      document.title = "SanityCheck Settings";
      themeSanityRadio.checked = true;
    }
    const stored = await chrome.storage.local.get(["customPrompt"]);
    if (stored.customPrompt) {
      promptEditor.value = stored.customPrompt;
      updateBadge(true);
    } else {
      promptEditor.value = DEFAULT_ANALYSIS_PROMPT;
      updateBadge(false);
    }
    savePromptBtn.addEventListener("click", () => {
      void savePrompt();
    });
    resetPromptBtn.addEventListener("click", () => {
      void resetPrompt();
    });
    themeSanityRadio.addEventListener("change", () => {
      void setTheme("sanity");
    });
    themeMissRadio.addEventListener("change", () => {
      void setTheme("miss");
    });
    backBtn.addEventListener("click", (e) => {
      e.preventDefault();
      window.close();
    });
    promptEditor.addEventListener("input", () => {
      const isCustom = promptEditor.value.trim() !== DEFAULT_ANALYSIS_PROMPT.trim();
      updateBadge(isCustom);
    });
  }
  function updateBadge(isCustom) {
    if (isCustom) {
      promptBadge.textContent = "Custom";
      promptBadge.classList.remove("default");
    } else {
      promptBadge.textContent = "Default";
      promptBadge.classList.add("default");
    }
  }
  async function savePrompt() {
    const promptText = promptEditor.value.trim();
    if (!promptText) {
      showStatus("Prompt cannot be empty", "error");
      return;
    }
    const isCustom = promptText !== DEFAULT_ANALYSIS_PROMPT.trim();
    if (isCustom) {
      await chrome.storage.local.set({ customPrompt: promptText });
    } else {
      await chrome.storage.local.remove(["customPrompt"]);
    }
    updateBadge(isCustom);
    showStatus("Prompt saved successfully!", "success");
  }
  async function resetPrompt() {
    promptEditor.value = DEFAULT_ANALYSIS_PROMPT;
    await chrome.storage.local.remove(["customPrompt"]);
    updateBadge(false);
    showStatus("Prompt reset to default", "success");
  }
  function showStatus(message, type) {
    statusMessage.textContent = message;
    statusMessage.className = `status-message ${type}`;
    setTimeout(() => {
      statusMessage.className = "status-message";
    }, 3e3);
  }
  async function setTheme(theme) {
    const body = document.body;
    if (theme === "miss") {
      body.classList.add("theme-miss");
      document.title = "Miss Information Settings";
    } else {
      body.classList.remove("theme-miss");
      document.title = "SanityCheck Settings";
    }
    await chrome.storage.local.set({ theme });
  }
})();
