"use strict";
(() => {
  // src/shared/colors.ts
  var baseColors = {
    // Status colors (same in both themes)
    error: "#ef4444",
    errorSubtle: "rgba(239, 68, 68, 0.12)",
    success: "#10b981",
    successSubtle: "rgba(16, 185, 129, 0.12)",
    warning: "#f59e0b",
    warningSubtle: "rgba(245, 158, 11, 0.12)",
    // Severity colors (same in both themes)
    severityCritical: "#ef4444",
    severitySignificant: "#eab308",
    severityMinor: "#737373"
  };
  var colors = {
    ...baseColors,
    // Backgrounds
    bgPrimary: "#0f1117",
    bgSecondary: "#1a1d27",
    bgTertiary: "#242936",
    bgHover: "#2d3344",
    // Text
    textPrimary: "#f0f2f5",
    textSecondary: "#9ca3b0",
    textMuted: "#6b7280",
    // Accent (blue)
    accent: "#60a5fa",
    accentHover: "#93c5fd",
    accentSubtle: "rgba(96, 165, 250, 0.12)",
    // Borders
    border: "rgba(255, 255, 255, 0.08)",
    borderStrong: "rgba(255, 255, 255, 0.12)",
    // Highlight colors (for reasoning gaps)
    highlightCritical: "rgba(239, 68, 68, 0.25)",
    highlightCriticalHover: "rgba(239, 68, 68, 0.4)",
    highlightSignificant: "rgba(234, 179, 8, 0.25)",
    highlightSignificantHover: "rgba(234, 179, 8, 0.4)",
    highlightMinor: "rgba(115, 115, 115, 0.25)",
    highlightMinorHover: "rgba(115, 115, 115, 0.4)",
    highlightDefault: "rgba(96, 165, 250, 0.25)",
    highlightDefaultHover: "rgba(96, 165, 250, 0.4)",
    severityDefault: "#60a5fa"
  };
  var missColors = {
    ...baseColors,
    // Backgrounds (purple-tinted dark)
    bgPrimary: "#1a1520",
    bgSecondary: "#231d2b",
    bgTertiary: "#2d2638",
    bgHover: "#3a3245",
    // Text (lavender-tinted)
    textPrimary: "#f5f0fa",
    textSecondary: "#c4b8d4",
    textMuted: "#8b7fa3",
    // Accent (purple)
    accent: "#c084fc",
    accentHover: "#d8b4fe",
    accentSubtle: "rgba(192, 132, 252, 0.12)",
    // Borders (purple-tinted)
    border: "rgba(192, 132, 252, 0.15)",
    borderStrong: "rgba(192, 132, 252, 0.25)",
    // Highlight colors (purple-shifted for theme consistency)
    highlightCritical: "rgba(239, 68, 68, 0.25)",
    highlightCriticalHover: "rgba(239, 68, 68, 0.4)",
    highlightSignificant: "rgba(234, 179, 8, 0.25)",
    highlightSignificantHover: "rgba(234, 179, 8, 0.4)",
    highlightMinor: "rgba(139, 127, 163, 0.25)",
    highlightMinorHover: "rgba(139, 127, 163, 0.4)",
    highlightDefault: "rgba(192, 132, 252, 0.25)",
    highlightDefaultHover: "rgba(192, 132, 252, 0.4)",
    severityDefault: "#c084fc"
  };
  function generateCssVariables(theme) {
    return `
  --bg-primary: ${theme.bgPrimary};
  --bg-secondary: ${theme.bgSecondary};
  --bg-tertiary: ${theme.bgTertiary};
  --bg-hover: ${theme.bgHover};
  --text-primary: ${theme.textPrimary};
  --text-secondary: ${theme.textSecondary};
  --text-muted: ${theme.textMuted};
  --accent: ${theme.accent};
  --accent-hover: ${theme.accentHover};
  --accent-subtle: ${theme.accentSubtle};
  --error: ${theme.error};
  --error-subtle: ${theme.errorSubtle};
  --success: ${theme.success};
  --success-subtle: ${theme.successSubtle};
  --warning: ${theme.warning};
  --warning-subtle: ${theme.warningSubtle};
  --border: ${theme.border};
  --border-strong: ${theme.borderStrong};
  --severity-critical: ${theme.severityCritical};
  --severity-significant: ${theme.severitySignificant};
  --severity-minor: ${theme.severityMinor};`;
  }
  var cssVariables = `
:root {
${generateCssVariables(colors)}
  --shadow: 0 4px 12px rgba(0, 0, 0, 0.25);
  --radius-sm: 6px;
  --radius: 10px;
  --radius-lg: 14px;
}
`;
  var missCssVariables = `
body.theme-miss {
${generateCssVariables(missColors)}
}
`;
  function missBackgroundCss(imagePath) {
    return `
body.theme-miss {
  background: 
    linear-gradient(
      to bottom,
      rgba(26, 21, 32, 0.85) 0%,
      rgba(26, 21, 32, 0.75) 50%,
      rgba(26, 21, 32, 0.9) 100%
    ),
    url('${imagePath}');
  background-size: cover;
  background-position: center;
  background-attachment: fixed;
}
`;
  }
  var themeCssVariables = cssVariables + missCssVariables + missBackgroundCss("/static/missinfo_bg.jpg");
  var themeCssVariablesExtension = cssVariables + missCssVariables + missBackgroundCss("icons/missinfo_bg.jpg");

  // src/shared/highlight-styles.ts
  var highlightCSS = `
/* ===== CSS Custom Highlight API Styles ===== */
::highlight(logic-checker-critical) {
  background-color: ${colors.highlightCritical};
}

::highlight(logic-checker-significant) {
  background-color: ${colors.highlightSignificant};
}

::highlight(logic-checker-minor) {
  background-color: ${colors.highlightMinor};
}

::highlight(logic-checker-default) {
  background-color: ${colors.highlightDefault};
}

/* ===== Fallback: Span-based Highlight Styles ===== */
.logic-checker-highlight {
  cursor: help;
  position: relative;
  transition: background 0.2s ease;
  border-radius: 2px;
  padding: 1px 2px;
}

.logic-checker-highlight.critical,
.logic-checker-highlight[data-importance="critical"] {
  background: linear-gradient(to bottom, ${colors.highlightCritical} 0%, rgba(239, 68, 68, 0.15) 100%);
}

.logic-checker-highlight.critical:hover,
.logic-checker-highlight[data-importance="critical"]:hover {
  background: ${colors.highlightCriticalHover};
}

.logic-checker-highlight.significant,
.logic-checker-highlight[data-importance="significant"] {
  background: linear-gradient(to bottom, ${colors.highlightSignificant} 0%, rgba(234, 179, 8, 0.15) 100%);
}

.logic-checker-highlight.significant:hover,
.logic-checker-highlight[data-importance="significant"]:hover {
  background: ${colors.highlightSignificantHover};
}

.logic-checker-highlight.minor,
.logic-checker-highlight[data-importance="minor"] {
  background: linear-gradient(to bottom, ${colors.highlightMinor} 0%, rgba(115, 115, 115, 0.15) 100%);
}

.logic-checker-highlight.minor:hover,
.logic-checker-highlight[data-importance="minor"]:hover {
  background: ${colors.highlightMinorHover};
}

.logic-checker-highlight:not(.critical):not(.significant):not(.minor):not([data-importance]) {
  background: linear-gradient(to bottom, ${colors.highlightDefault} 0%, rgba(96, 165, 250, 0.15) 100%);
}

.logic-checker-highlight:not(.critical):not(.significant):not(.minor):not([data-importance]):hover {
  background: ${colors.highlightDefaultHover};
}
`;
  var tooltipCSS = `
.logic-checker-tooltip {
  position: fixed;
  z-index: 2147483647;
  width: max-content;
  max-width: 400px;
  min-width: 280px;
  background: linear-gradient(135deg, #1a1a1a 0%, #0d0d0d 100%);
  border: 1px solid;
  border-radius: 8px;
  padding: 12px 14px;
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5);
  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  font-size: 13px;
  line-height: 1.5;
  color: #f5f5f5;
  pointer-events: none;
  opacity: 0;
  transform: translateY(8px);
  transition: opacity 0.2s ease, transform 0.2s ease;
}

.logic-checker-tooltip.critical {
  border-color: ${colors.severityCritical};
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(239, 68, 68, 0.2);
}

.logic-checker-tooltip.significant {
  border-color: ${colors.severitySignificant};
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(234, 179, 8, 0.2);
}

.logic-checker-tooltip.minor {
  border-color: ${colors.severityMinor};
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(115, 115, 115, 0.2);
}

.logic-checker-tooltip:not(.critical):not(.significant):not(.minor) {
  border-color: ${colors.severityDefault};
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(96, 165, 250, 0.2);
}

.logic-checker-tooltip.visible {
  opacity: 1;
  transform: translateY(0);
}

.logic-checker-tooltip-header {
  margin-bottom: 6px;
}

.logic-checker-tooltip-badge {
  display: inline-block;
  color: #000;
  font-size: 10px;
  font-weight: 600;
  padding: 2px 8px;
  border-radius: 10px;
  text-transform: uppercase;
  letter-spacing: 0.5px;
}

.logic-checker-tooltip.critical .logic-checker-tooltip-badge {
  background: ${colors.severityCritical};
  color: white;
}

.logic-checker-tooltip.significant .logic-checker-tooltip-badge {
  background: ${colors.severitySignificant};
}

.logic-checker-tooltip.minor .logic-checker-tooltip-badge {
  background: ${colors.severityMinor};
  color: white;
}

.logic-checker-tooltip:not(.critical):not(.significant):not(.minor) .logic-checker-tooltip-badge {
  background: ${colors.severityDefault};
}

.logic-checker-tooltip-explanation {
  color: #d4d4d4;
}
`;
  var contentStyles = highlightCSS + "\n" + tooltipCSS;
  var demoHighlightCSS = `
.demo-highlight {
  cursor: help;
  position: relative;
  background: linear-gradient(to bottom, ${colors.highlightSignificant} 0%, rgba(234, 179, 8, 0.15) 100%);
  border-radius: 2px;
  padding: 1px 2px;
  transition: background 0.2s ease;
}

.demo-highlight:hover {
  background: ${colors.highlightSignificantHover};
}

.demo-highlight.critical {
  background: linear-gradient(to bottom, ${colors.highlightCritical} 0%, rgba(239, 68, 68, 0.15) 100%);
}

.demo-highlight.critical:hover {
  background: ${colors.highlightCriticalHover};
}

.demo-highlight.minor {
  background: linear-gradient(to bottom, ${colors.highlightMinor} 0%, rgba(115, 115, 115, 0.15) 100%);
}

.demo-highlight.minor:hover {
  background: ${colors.highlightMinorHover};
}

.demo-tooltip {
  position: absolute;
  z-index: 1000;
  width: max-content;
  max-width: 400px;
  min-width: 280px;
  background: linear-gradient(135deg, #1a1a1a 0%, #0d0d0d 100%);
  border: 1px solid ${colors.severitySignificant};
  border-radius: 8px;
  padding: 12px 14px;
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(234, 179, 8, 0.2);
  font-size: 13px;
  line-height: 1.5;
  color: #f5f5f5;
  pointer-events: none;
  opacity: 0;
  transform: translateY(8px);
  transition: opacity 0.2s ease, transform 0.2s ease;
  left: 0;
  top: 100%;
  margin-top: 8px;
}

.demo-highlight:hover .demo-tooltip {
  opacity: 1;
  transform: translateY(0);
}

.demo-tooltip-header {
  margin-bottom: 6px;
}

.demo-tooltip-badge {
  display: inline-block;
  background: ${colors.severitySignificant};
  color: #000;
  font-size: 10px;
  font-weight: 600;
  padding: 2px 8px;
  border-radius: 10px;
  text-transform: uppercase;
  letter-spacing: 0.5px;
}

.demo-tooltip-explanation {
  color: #d4d4d4;
}
`;

  // src/shared/kawaii.ts
  function makeKawaii(text) {
    if (!text) return text;
    let result = text;
    result = result.replace(/\.(\s+|$)/g, (match, space) => {
      if (Math.random() < 0.6) {
        return "~" + space;
      }
      return match;
    });
    const loveWords = ["good", "great", "excellent", "helpful", "useful", "important", "nice", "correct", "right", "valid"];
    loveWords.forEach((word) => {
      const regex = new RegExp(`\\b${word}\\b`, "gi");
      result = result.replace(regex, (match) => {
        if (Math.random() < 0.4) {
          return match + " <3";
        }
        return match;
      });
    });
    if (result.length > 30) {
      result = result.replace(/([.!?])(\s+)/g, (match, punct, space) => {
        if (Math.random() < 0.5) {
          const kawaiis = ["uwu", "owo", ">w<", "nya~"];
          const chosen = kawaiis[Math.floor(Math.random() * kawaiis.length)];
          return punct + " " + chosen + space;
        }
        return match;
      });
    }
    result = result.replace(/([.!?])(\s+)/g, (match, punct, space) => {
      if (Math.random() < 0.25) {
        return punct + " \u2728" + space;
      }
      return match;
    });
    if (Math.random() < 0.3) {
      const emoticons = [" (\u25D5\u203F\u25D5)", " (\xB4\u2200\uFF40)", " (\u0E51\u2022\u0300\u03C9\u2022\u0301\u0E51)", " (\u25E1\u203F\u25E1)", " (\uFF61\u25D5\u203F\u25D5\uFF61)"];
      const chosen = emoticons[Math.floor(Math.random() * emoticons.length)];
      result = result.replace(/([.!?])(\s*)$/, "$1" + chosen + "$2");
    }
    result = result.replace(/it is important to note/gi, (match) => Math.random() < 0.7 ? "just so you know~" : match);
    result = result.replace(/it should be noted/gi, (match) => Math.random() < 0.7 ? "heads up~" : match);
    result = result.replace(/it is worth noting/gi, (match) => Math.random() < 0.7 ? "worth mentioning~" : match);
    result = result.replace(/this suggests/gi, (match) => Math.random() < 0.6 ? "this kinda suggests" : match);
    result = result.replace(/this indicates/gi, (match) => Math.random() < 0.6 ? "this kinda indicates" : match);
    result = result.replace(/\bhowever\b/gi, (match, offset) => offset > 0 && Math.random() < 0.7 ? "but" : match);
    result = result.replace(/\bfurthermore\b/gi, (match) => Math.random() < 0.6 ? "also~" : match);
    result = result.replace(/\btherefore\b/gi, (match) => Math.random() < 0.6 ? "so" : match);
    result = result.replace(/\bmoreover\b/gi, (match) => Math.random() < 0.5 ? "plus~" : match);
    result = result.replace(/\bconsequently\b/gi, (match) => Math.random() < 0.5 ? "so" : match);
    result = result.replace(/\bnevertheless\b/gi, (match) => Math.random() < 0.5 ? "but still~" : match);
    if (Math.random() < 0.2) {
      const kawaiiPhrases = [" uwu", " owo", " nya~", " >w<", " (\xB4\uFF61\u2022 \u1D55 \u2022\uFF61`) \u2661"];
      const chosen = kawaiiPhrases[Math.floor(Math.random() * kawaiiPhrases.length)];
      const words = result.split(" ");
      if (words.length > 3 && chosen) {
        const insertAt = Math.floor(words.length / 2) + Math.floor(Math.random() * 3) - 1;
        if (words[insertAt]) {
          words[insertAt] += chosen;
          result = words.join(" ");
        }
      }
    }
    return result;
  }

  // src/extension/welcome.ts
  var articles = [
    {
      url: "healthnews.com/coffee-longevity-study",
      content: `
      <p>
        A new study of 180,000 adults found that those who drank 3-4 cups of coffee daily had a 15% lower risk of death over the 12-year follow-up period. <span class="highlight significant" data-type="Hidden Confounders" data-importance="significant" data-explanation="Standard controls miss key confounders like socioeconomic status, sleep quality, and pre-existing conditions. Coffee drinkers may differ from non-drinkers in unmeasured ways that actually explain the longevity difference.">The researchers controlled for smoking, alcohol, and exercise.</span> The findings suggest coffee could be a key to living longer.
      </p>
      <p>
        Lead author Dr. Martinez notes that coffee's antioxidant properties may explain the protective effect. <span class="highlight minor" data-type="Publication Bias" data-importance="minor" data-explanation="Studies finding no effect are less likely to be published. Consistent positive findings across studies may reflect shared methodological biases rather than genuine truth.">Previous studies have shown similar associations,</span> making this one of the most robust findings in nutritional science.
      </p>
      <p>
        Interestingly, the effect was strongest among participants who had been drinking coffee for decades. <span class="highlight critical" data-type="Reverse Causation" data-importance="critical" data-explanation="Long-term coffee drinkers still alive may have been healthier to begin with. Sicker people often quit coffee on medical advice, creating the false appearance of a dose-response relationship.">This suggests the benefits compound over time,</span> so starting early may be especially important.
      </p>
    `
    },
    {
      url: "techinsider.com/unicorn-patterns",
      content: `
      <p>
        After analyzing 500 successful startups, we identified three common traits: strong founding teams, rapid iteration, and deep customer empathy. <span class="highlight critical" data-type="Base Rate Neglect" data-importance="critical" data-explanation="Without knowing how common these traits are in failed startups, 87% tells us nothing. If 90% of failures also had these traits, they're not predictive of success at all.">These factors were present in 87% of companies that reached unicorn status.</span>
      </p>
      <p>
        What's remarkable is how consistent the pattern is. <span class="highlight significant" data-type="Narrative Fallacy" data-importance="significant" data-explanation="Post-hoc analysis finds patterns because humans are pattern-seekers. The 'playbook' is constructed backward from success\u2014their paths likely looked very different in real-time, with luck playing a major role.">From Stripe to Notion to Figma, the same playbook appears again and again.</span> This suggests a replicable formula for startup success.
      </p>
      <p>
        Notably, many of these founders had no formal business training. <span class="highlight significant" data-type="Survivorship Bias" data-importance="significant" data-explanation="We're not hearing from founders with intuition and hustle who failed. Credentials may reduce variance in outcomes even if some succeed without them\u2014we just don't see the failures.">Their success came from intuition and hustle rather than MBA frameworks</span>\u2014a lesson for aspiring entrepreneurs.
      </p>
    `
    },
    {
      url: "wealthwise.substack.com/long-term-investing",
      content: `
      <p>
        The data is clear: over any 20-year period in U.S. history, the stock market has recovered from every crash. <span class="highlight critical" data-type="Survivorship Bias in Markets" data-importance="critical" data-explanation="We study markets that recovered. Markets that went to zero (Russia 1917, China 1949, Germany 1945) aren't in our dataset. The U.S. surviving and thriving isn't guaranteed\u2014it's one draw from possible histories.">This makes staying invested through downturns the only rational strategy.</span>
      </p>
      <p>
        Dollar-cost averaging removes the stress of timing. <span class="highlight significant" data-type="Mathematically Misleading" data-importance="significant" data-explanation="This sounds like a timing advantage, but it isn't. DCA actually underperforms lump-sum investing roughly 68% of the time. The 'buy more when low' framing obscures that you're just slowly entering a rising market.">By investing the same amount monthly, you automatically buy more shares when prices are low and fewer when high.</span>
      </p>
      <p>
        The math speaks for itself: the S&P 500 has returned roughly 10% annually over the past century. <span class="highlight significant" data-type="Geometric vs. Arithmetic Mean" data-importance="significant" data-explanation="10% average annual returns \u2260 10% compound growth. Volatility drag means actual wealth grows slower than the average suggests. A portfolio gaining 50% then losing 33% 'averages' 8.5% annually but has returned exactly 0%.">Even accounting for inflation, that's 7% real returns\u2014enough to double your money every decade.</span>
      </p>
    `
    }
  ];
  var currentArticleIndex = 0;
  var tooltip = document.getElementById("tooltip");
  var fakeCursor = document.getElementById("fake-cursor");
  var browserContent = document.getElementById("browser-content");
  var browserUrl = document.getElementById("browser-url");
  var articleText = document.getElementById("article-text");
  var pageIndicators = document.getElementById("page-indicators");
  var prevBtn = document.getElementById("prev-btn");
  var nextBtn = document.getElementById("next-btn");
  var highlights = [];
  var isUserControlling = false;
  var animationPaused = false;
  var currentHighlightIndex = 0;
  var animationTimeout = null;
  var cursorX = 50;
  var cursorY = 50;
  var animationFrameId = null;
  function initPageIndicators() {
    pageIndicators.innerHTML = "";
    articles.forEach((_, i) => {
      const dot = document.createElement("div");
      dot.className = `page-dot ${i === currentArticleIndex ? "active" : ""}`;
      dot.addEventListener("click", () => goToArticle(i));
      pageIndicators.appendChild(dot);
    });
  }
  function updatePageIndicators() {
    const dots = pageIndicators.querySelectorAll(".page-dot");
    dots.forEach((dot, i) => {
      dot.classList.toggle("active", i === currentArticleIndex);
    });
  }
  function loadArticle(index) {
    const article = articles[index];
    if (!article) return;
    browserUrl.textContent = article.url;
    articleText.innerHTML = article.content;
    highlights = Array.from(document.querySelectorAll(".highlight"));
    highlights.forEach((highlight) => {
      highlight.addEventListener("mouseenter", (e) => {
        if (!isUserControlling) return;
        showTooltipForHighlight(highlight, e.clientX, e.clientY);
      });
      highlight.addEventListener("mouseleave", () => {
        if (!isUserControlling) return;
        hideTooltip();
      });
      highlight.addEventListener("mousemove", (e) => {
        if (!isUserControlling) return;
        positionTooltipAt(e.clientX, e.clientY);
      });
    });
    updatePageIndicators();
  }
  function goToArticle(index) {
    stopAnimation();
    currentArticleIndex = index;
    currentHighlightIndex = 0;
    cursorX = 50;
    cursorY = 50;
    fakeCursor.style.left = "50px";
    fakeCursor.style.top = "50px";
    loadArticle(index);
    setTimeout(startAnimation, 300);
  }
  function nextArticle() {
    goToArticle((currentArticleIndex + 1) % articles.length);
  }
  function prevArticle() {
    goToArticle((currentArticleIndex - 1 + articles.length) % articles.length);
  }
  function showTooltipForHighlight(highlight, cursorXPos, cursorYPos) {
    const type = highlight.dataset.type || "Issue";
    const importance = highlight.dataset.importance || "minor";
    let explanation = highlight.dataset.explanation || "";
    if (document.body.classList.contains("theme-miss")) {
      explanation = makeKawaii(explanation);
    }
    const emoji = importance === "critical" ? "\u{1F534}" : importance === "significant" ? "\u{1F7E0}" : "\u{1F7E1}";
    tooltip.className = `tooltip ${importance}`;
    tooltip.innerHTML = `
    <div class="tooltip-badge">Logic Issue</div>
    <div class="tooltip-header">
      <span class="tooltip-icon">${emoji}</span>
      <span class="tooltip-type">${type}</span>
    </div>
    <div class="tooltip-text">${explanation}</div>
  `;
    positionTooltipAt(cursorXPos, cursorYPos);
    tooltip.classList.add("visible");
    highlight.classList.add("active");
  }
  function hideTooltip() {
    tooltip.classList.remove("visible");
    highlights.forEach((h) => h.classList.remove("active"));
  }
  function positionTooltipAt(x, y) {
    const padding = 15;
    const rect = tooltip.getBoundingClientRect();
    let left = x + padding;
    let top = y + padding;
    if (left + rect.width > window.innerWidth - padding) {
      left = x - rect.width - padding;
    }
    if (top + rect.height > window.innerHeight - padding) {
      top = y - rect.height - padding;
    }
    tooltip.style.left = `${Math.max(padding, left)}px`;
    tooltip.style.top = `${Math.max(padding, top)}px`;
  }
  function quadraticBezier(t, p0, p1, p2) {
    const mt = 1 - t;
    return mt * mt * p0 + 2 * mt * t * p1 + t * t * p2;
  }
  function animateCursorToPosition(targetX, targetY, duration, callback) {
    const startX = cursorX;
    const startY = cursorY;
    const startTime = performance.now();
    const midX = (startX + targetX) / 2;
    const midY = (startY + targetY) / 2;
    const distance = Math.sqrt((targetX - startX) ** 2 + (targetY - startY) ** 2);
    const angle = Math.atan2(targetY - startY, targetX - startX) + Math.PI / 2;
    const arcIntensity = (Math.random() - 0.5) * distance * 0.8;
    const controlX = midX + Math.cos(angle) * arcIntensity;
    const controlY = midY + Math.sin(angle) * arcIntensity;
    function animate(currentTime) {
      if (isUserControlling || animationPaused) {
        if (animationFrameId) cancelAnimationFrame(animationFrameId);
        return;
      }
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);
      const eased = 1 - Math.pow(1 - progress, 3);
      cursorX = quadraticBezier(eased, startX, controlX, targetX);
      cursorY = quadraticBezier(eased, startY, controlY, targetY);
      fakeCursor.style.left = `${cursorX}px`;
      fakeCursor.style.top = `${cursorY}px`;
      if (progress < 1) {
        animationFrameId = requestAnimationFrame(animate);
      } else {
        if (callback) callback();
      }
    }
    animationFrameId = requestAnimationFrame(animate);
  }
  function animateCursor() {
    if (isUserControlling || animationPaused) return;
    if (highlights.length === 0) return;
    const highlight = highlights[currentHighlightIndex];
    if (!highlight) return;
    const contentRect = browserContent.getBoundingClientRect();
    const highlightRect = highlight.getBoundingClientRect();
    const xOffset = 0.15 + Math.random() * 0.35;
    const yOffset = 0.3 + Math.random() * 0.4;
    const targetX = highlightRect.left + highlightRect.width * xOffset - contentRect.left;
    const targetY = highlightRect.top + highlightRect.height * yOffset - contentRect.top;
    animateCursorToPosition(targetX, targetY, 800, () => {
      if (isUserControlling || animationPaused) return;
      const absX = highlightRect.left + highlightRect.width * xOffset;
      const absY = highlightRect.top + highlightRect.height * yOffset;
      showTooltipForHighlight(highlight, absX, absY);
      animationTimeout = setTimeout(() => {
        if (isUserControlling || animationPaused) return;
        hideTooltip();
        animationTimeout = setTimeout(() => {
          if (isUserControlling || animationPaused) return;
          currentHighlightIndex = (currentHighlightIndex + 1) % highlights.length;
          animateCursor();
        }, 300);
      }, 2500);
    });
  }
  function startAnimation() {
    isUserControlling = false;
    animationPaused = false;
    fakeCursor.classList.remove("hidden");
    animateCursor();
  }
  function stopAnimation() {
    isUserControlling = true;
    if (animationTimeout) {
      clearTimeout(animationTimeout);
      animationTimeout = null;
    }
    if (animationFrameId) {
      cancelAnimationFrame(animationFrameId);
      animationFrameId = null;
    }
    fakeCursor.classList.add("hidden");
    hideTooltip();
  }
  browserContent.addEventListener("mouseenter", () => {
    stopAnimation();
  });
  browserContent.addEventListener("mouseleave", () => {
    hideTooltip();
    setTimeout(() => {
      if (!isUserControlling) return;
      startAnimation();
    }, 1e3);
  });
  prevBtn.addEventListener("click", prevArticle);
  nextBtn.addEventListener("click", nextArticle);
  var settingsLink = document.getElementById("settings-link");
  if (settingsLink) {
    settingsLink.addEventListener("click", (e) => {
      e.preventDefault();
      if (typeof chrome !== "undefined" && chrome.tabs) {
        chrome.tabs.create({ url: chrome.runtime.getURL("settings.html") });
      } else {
        window.location.href = "settings.html";
      }
    });
  }
  initPageIndicators();
  loadArticle(0);
  setTimeout(startAnimation, 500);
  var themeToggle = document.getElementById("theme-toggle");
  var labelSanity = document.getElementById("label-sanity");
  var labelMiss = document.getElementById("label-miss");
  function toggleTheme() {
    document.body.classList.toggle("theme-miss");
    themeToggle.classList.toggle("active");
    const isMiss = document.body.classList.contains("theme-miss");
    labelSanity.classList.toggle("active", !isMiss);
    labelMiss.classList.toggle("active", isMiss);
    document.title = isMiss ? "Welcome to Miss Information \u2661" : "Welcome to SanityCheck";
    localStorage.setItem("theme", isMiss ? "miss" : "sanity");
    if (typeof chrome !== "undefined" && chrome.storage) {
      chrome.storage.local.set({ theme: isMiss ? "miss" : "sanity" });
    }
  }
  themeToggle.addEventListener("click", toggleTheme);
  if (typeof chrome !== "undefined" && chrome.storage) {
    chrome.storage.local.get(["theme"], (result) => {
      const savedTheme = result.theme || localStorage.getItem("theme");
      if (savedTheme === "miss" && !document.body.classList.contains("theme-miss")) {
        toggleTheme();
      }
    });
  } else {
    if (localStorage.getItem("theme") === "miss") {
      toggleTheme();
    }
  }
  (function() {
    const container = document.getElementById("how-rotating-container");
    if (!container) return;
    const steps = document.querySelectorAll(".how-step[data-step]");
    const dots = document.querySelectorAll(".how-nav-dot[data-step]");
    let currentStep = 0;
    let stepInterval = null;
    let isHovered = false;
    const FULL_ROTATION_TIME = 2e4;
    const STEP_TIME = FULL_ROTATION_TIME / 4;
    const HOVER_PADDING = 50;
    function showStep(stepIndex) {
      steps.forEach((step, index) => {
        step.classList.remove("active", "prev");
        if (index === stepIndex) {
          step.classList.add("active");
        } else if (index < stepIndex) {
          step.classList.add("prev");
        }
      });
      dots.forEach((dot, index) => {
        dot.classList.toggle("active", index === stepIndex);
      });
      currentStep = stepIndex;
    }
    function nextStep() {
      const next = (currentStep + 1) % steps.length;
      showStep(next);
    }
    function startStepRotation() {
      if (stepInterval) clearInterval(stepInterval);
      if (!isHovered) {
        stepInterval = setInterval(nextStep, STEP_TIME);
      }
    }
    function stopStepRotation() {
      if (stepInterval) {
        clearInterval(stepInterval);
        stepInterval = null;
      }
      container.classList.add("paused");
    }
    function checkMouseProximity(e) {
      const rect = container.getBoundingClientRect();
      const padding = HOVER_PADDING;
      const x = e.clientX;
      const y = e.clientY;
      const isNear = x >= rect.left - padding && x <= rect.right + padding && y >= rect.top - padding && y <= rect.bottom + padding;
      if (isNear && !isHovered) {
        isHovered = true;
        stopStepRotation();
      } else if (!isNear && isHovered) {
        isHovered = false;
        container.classList.remove("paused");
        startStepRotation();
      }
    }
    dots.forEach((dot, index) => {
      dot.addEventListener("click", () => {
        stopStepRotation();
        showStep(index);
        setTimeout(() => {
          if (!isHovered) {
            container.classList.remove("paused");
            startStepRotation();
          }
        }, 1e3);
      });
    });
    document.addEventListener("mousemove", checkMouseProximity);
    showStep(0);
    startStepRotation();
  })();
})();
